import os
import hashlib
import requests
import time
from email import policy
from email.parser import BytesParser
import argparse

# Configuration des clés API
CLE_API_VIRUSTOTAL = 'API'  
CLE_API_MALWAREBAZAAR = 'API'  

URL_VIRUSTOTAL = 'https://www.virustotal.com/api/v3/files/'
URL_MALWAREBAZAAR = 'https://mb-api.abuse.ch/api/v1/'
ENTETES_VIRUSTOTAL = {'x-apikey': CLE_API_VIRUSTOTAL}

def calculer_sha256(data):
    sha256 = hashlib.sha256(data).hexdigest()
    return sha256

def verifier_virustotal(sha256):
    try:
        reponse = requests.get(URL_VIRUSTOTAL + sha256, headers=ENTETES_VIRUSTOTAL)
        if reponse.status_code == 200:
            donnees = reponse.json()
            stats = donnees['data']['attributes']['last_analysis_stats']
            return stats['malicious'], stats['harmless']
        return None, None
    except Exception as e:
        print(f"Erreur VirusTotal: {str(e)}")
        return None, None

def verifier_malwarebazaar(sha256):
    try:
        donnees = {
            'query': 'get_info',
            'hash': sha256,
            'apikey': CLE_API_MALWAREBAZAAR
        }
        reponse = requests.post(URL_MALWAREBAZAAR, data=donnees)
        if reponse.status_code == 200:
            resultat = reponse.json()
            return resultat.get('query_status') == 'ok'
        return False
    except Exception as e:
        print(f"Erreur MalwareBazaar: {str(e)}")
        return False

def extraire_pieces_jointes(eml_file):
    pieces_jointes = []
    with open(eml_file, 'rb') as f:
        msg = BytesParser(policy=policy.default).parse(f)
        for part in msg.iter_attachments():
            filename = part.get_filename()
            if filename:
                data = part.get_payload(decode=True)
                pieces_jointes.append((filename, data))
    return pieces_jointes

def analyser_dossier_eml(dossier):
    for fichier in os.listdir(dossier):
        if fichier.lower().endswith('.eml'):
            chemin_eml = os.path.join(dossier, fichier)
            try:
                print(f"\n📧 Traitement du fichier EML : {fichier}")
                pieces = extraire_pieces_jointes(chemin_eml)
                
                if not pieces:
                    print("Aucune pièce jointe trouvée")
                    continue
                
                for nom_fichier, data in pieces:
                    print(f"\n📎 Pièce jointe : {nom_fichier}")
                    sha256 = calculer_sha256(data)
                    print(f"🔒 SHA-256 : {sha256}")
                    
                    malveillant, inoffensif = verifier_virustotal(sha256)
                    if malveillant is not None:
                        if malveillant > 0:
                            print(f"🛑 VirusTotal : {malveillant} détections malveillantes")
                        else:
                            print(f"✅ VirusTotal : {inoffensif} détections inoffensives")
                    
                    if verifier_malwarebazaar(sha256):
                        print("🛑 MalwareBazaar : Signature malveillante détectée")
                    else:
                        print("✅ MalwareBazaar : Aucun résultat trouvé")
                    
                    time.sleep(15) 

            except Exception as e:
                print(f"❌ Erreur avec le fichier {fichier} : {str(e)}")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Analyse les pièces jointes EML')
    parser.add_argument('dossier', help='Dossier contenant les fichiers EML')
    args = parser.parse_args()

    if not os.path.isdir(args.dossier):
        print("❌ Le chemin spécifié n'est pas un dossier valide")
    else:
        analyser_dossier_eml(args.dossier)